
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  main.c: SD card bootloader
  Written by Nicholas Vinen, 2010-2013
*/

#include "p33Fxxxx.h"
#include "../sdcard/diskio.h"
#include "../sdcard/ff.h"
#include "../sdcard/spi.h"
#include "hexfile.h"
#include "flash_operations.h"
#include <string.h>

/*
  chip: dsPIC33FJ128GP306
  clock: 7.37MHz [FRC]/39.6MHz [FRCPLL]
  pins:

    RB0  (16): LED1 1.2k cathode
    RB1  (15): LED1 680R cathode
    RB2  (14): LED2 1.2k cathode
    RB3  (13): LED2 680R cathode
    RB4  (12): LED3 1.2k cathode
    RB5  (11): LED3 680R cathode
    RB6  (17): LED4 820R cathode
    RB7  (18): LED4 470R cathode
    RB8  (21): LED5 1.2k cathode
    RB9  (22): LED5 680R cathode
    RB10 (23): LED6 1.2k cathode
    RB11 (24): LED6 680R cathode
    RB12 (27): LED7 1.2k cathode
    RB13 (28): LED7 680R cathode
    RB14 (29): LED8 2.2k cathode
    RB15 (30): LED8 1.2k cathode

    RC1  ( 2): power control (active high) (controls +15V, -15V & TOSLINK supplies)
    RC2  ( 3): CS8416 CS
    RC12 (39): Sample rate LEDs anode #1 (??)
  * RC13 (47): CS8416 GPO2 [CN1]
    RC14 (48): CS4398 RST
    RC15 (40): Sample rate LEDs anode #2 (??)

    RD0  (46): PLL1708 CS
    RD1  (49): CS4398 CS
    RD2  (50): CS8416 RST
    RD3  (51): Headphone socket switch (closed = low)
  * RD4  (52): CS4398 LMUTEC
    RD5  (53): CS8416 GPO1 [CN14]
    RD6  (54): PCM2902 SSPND
    RD7  (55): Power switch/AC sense
    RD8  (42): Sample rate LEDs cathode 4.7k# (??)
    RD9  (43): Sample rate LEDs anode #3 (??)
    RD10 (44): Sample rate LEDs anode #4 (??)
    RD11 (45): Sample rate LEDs cathode 2.2k# (??)

    RF0  (58): DIP switch #1
    RF1  (59): DIP switch #2
    RF2  (34): SD card DATA OUT
    RF3  (33): SD card DATA IN
    RF4  (31): SD card detect (present = low) [CN17]
    RF5  (32): SD card CS
    RF6  (35): SD card CLK

    RG0  (61): Power LED (active low)
    RG1  (60): DIP switch #3
    RG2  (37): SD card +3.3V power enable (active low)
    RG3  (36): SD card write protect (protected = low)
    RG6  ( 4): Control bus serial clock [SCK2]
    RG7  ( 5): Control bus data input [SDI2]
    RG8  ( 6): Control bus serial data  [SDO2]
    RG9  ( 8): Infrared data input [CN11]
    RG12 (63): N/C
    RG13 (64): CS8416 RXP7 (Input #7) [CSDO]
    RG14 (62): PLL1708 SCKO2 / 2 [CSCK]
    RG15 ( 1): N/C

  * = unused
*/

// Configuration registers
_FGS(GWRP_OFF & GCP_OFF);                        // We don't want to write protect the flash as we use it to store state.
_FOSCSEL(FNOSC_FRC);                             // Start up initially without the PLL but instead using the internal fast RC oscillator.
_FOSC(FCKSM_CSECMD & OSCIOFNC_ON & POSCMD_NONE); // Clock switching enabled, no clock output, no crystal driver.
_FWDT(FWDTEN_OFF);                               // Not using the watchdog timer.
#ifdef __DEBUG
_FICD(ICS_PGD3&JTAGEN_OFF);                      // Debug on ICSP port.
#endif

//////////// Functions and variables for controlling power to SD card and determining SD card interface state ////////////

unsigned char memoryCardSystemUp;

static void SetSDCardPower(unsigned char bPowered) {
  DeinitSPI();
  LATGbits.LATG2 = !bPowered;
  TRISGbits.TRISG2 = 0;
}

static void InitSDCardDetect() {
  TRISFbits.TRISF4 = 1;
  CNPU2bits.CN17PUE = 1;
}

//////////// Simple delay function ////////////

void Timer_Delay_ms_escape(unsigned short ms, unsigned char* interrupt, unsigned char interrupt_cmp) {
	do {
		unsigned short delay = ms;
		if( delay > 400 )
			delay = 400;

		TMR1 = 0;
		PR1 = 77*delay;
		IFS0bits.T1IF = 0;
		T1CONbits.TON = 1;
		while( !IFS0bits.T1IF ) {
			memoryCardSystemUp = !PORTFbits.RF4;
			if( interrupt && *interrupt == interrupt_cmp )
				return;
		}
		T1CONbits.TON = 0;
		IFS0bits.T1IF = 0;

		ms -= delay;
	} while( ms );
}

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200

FATFS fs;
DIR cur_pos;
FILINFO cur_file_info;
FIL cur_file;

#define HEX_FILE_NAME "DAC.hex"

//////////// Restores microcontroller state to valid boot-time configuration then jumps to user code ////////////

static void boot_main_program() {
	// restore state to default

	// undo pull-up for card sense pin
    CNPU2bits.CN17PUE = 0;

	// reset SPI for SD card
	SPI1STAT = 0;
	SPI1CON1 = 0;
	SPI1CON2 = 0;

	_SPI1IF = 0;
    TRISF = 0xFFFF;

	// reset DMA for SD card
	DMA1CON = 0;
	DMA1STA = 0;
	DMA1PAD = 0;
	DMA1CNT = 0;
	DMA1REQ = 0;

	// reset LED stuff
    TRISB = 0xFFFF;
    TRISC = 0xFFFF;
    TRISD = 0xFFFF;

	// reset timer used for Timer_Delay_ms_escape
	T1CON = 0;
	PR1 = 0xFFFF;
	IFS0bits.T1IF = 0;

    // change back to FRC for clock (no PLL)
    __builtin_write_OSCCONH(0x00);
    __builtin_write_OSCCONL(0x01);

	__asm__ volatile("goto 0x200");
}

//////////// Functions to display bootloader progress on unit's front panel LEDs ////////////

inline static void init_leds() {
    LATB = 0;
	TRISDbits.TRISD10 = 0;
	TRISDbits.TRISD9 = 0;
	TRISCbits.TRISC15 = 0;
	TRISCbits.TRISC12 = 0;
	LATDbits.LATD11 = 1;
	LATDbits.LATD8  = 1;
	TRISDbits.TRISD11 = 0;
	TRISDbits.TRISD8 = 1;
}

inline static void set_led_state(unsigned char sampling_rate, unsigned char input) {
  unsigned short temp = 0;
  unsigned char i;
  for( i = 0; i < 8; ++i ) {
    temp >>= 2;
    if( !(input&1) )
      temp |= 0xC000;
    input >>= 1;
  }
  LATDbits.LATD10 = !(sampling_rate&1);
  LATDbits.LATD9  = !((sampling_rate>>1)&1);
  LATCbits.LATC15 = !((sampling_rate>>2)&1);
  LATCbits.LATC12 = !((sampling_rate>>3)&1);
  TRISB = temp;
}

static void error_flash(unsigned char long_flashes, unsigned char short_flashes, unsigned char* interrupt, unsigned char interrupt_cmp) {
	while(!interrupt || *interrupt != interrupt_cmp) {
		set_led_state(long_flashes, short_flashes);
		Timer_Delay_ms_escape(300, interrupt, interrupt_cmp);
		set_led_state(0, 0);
		Timer_Delay_ms_escape(300, interrupt, interrupt_cmp);
	}
	set_led_state(0, 0);
}

static void rapid_flash(unsigned short delay) {
	unsigned char i = 1, j = 1;
    while( i > 1 || j == 1 ) {
		set_led_state(i, 0);
		Timer_Delay_ms_escape(100, 0, 0);
		if( j == 1 ) {
			i <<= 1;
			if( i == 8 )
				j = 2;
		} else {
			i >>= 1;
		}
	}
	set_led_state(0, 0);
}

unsigned char progress[2];
unsigned long last_bytes_written, total_bytes_to_write;
void progress_callback(unsigned long bytes_written, unsigned char callback_mode) {
  if( bytes_written - last_bytes_written >= 4096 ) {
    last_bytes_written += 4096;
    if( progress[1] ) {
      if( --progress[0] == 0 )
        progress[1] = 0;
    } else {
      if( ++progress[0] == 3 )
        progress[1] = 1;
    }
  }
  if( callback_mode == 1 ) {
    set_led_state(1<<progress[0], 0);
  } else {
    unsigned char bargraph;

    bargraph = total_bytes_to_write ? bytes_written * 9 / total_bytes_to_write : 0;
    if( bargraph > 8 )
      bargraph = 8;
    set_led_state(1<<progress[0], ((1<<bargraph)-1)^(callback_mode == 2 ? 0 : 0xFF));
  }
}

//////////// Main bootloader functions below ////////////

static unsigned char init_sdcard_and_open_hex_file() {
	unsigned char i;

	for( i = 0; i < INIT_ATTEMPTS; ++i ) {
		unsigned char result;
		Timer_Delay_ms_escape(INIT_DELAY, &memoryCardSystemUp, 0);
		result = disk_initialize(0);
		if( result != STA_NOINIT )
			break;
	}
	if( i == INIT_ATTEMPTS )
		return 1;

	if( f_mount(0, &fs) != FR_OK )
		return 2;

	if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK )
		return 3;

	return 0;
}

unsigned long boot_address_cache[2];
void flash_block_erased(unsigned long addr) {
	if( addr == 0 ) {
		write_flash_word(0, boot_address_cache[0]);
		write_flash_word(2, boot_address_cache[1]);
	}
}

int main(void) {
	unsigned char error, ret;//, x = 1, y = 1;

	// set up and switch to PLL
	PLLFBD = 41;
	CLKDIVbits.PLLPRE = 0;
	CLKDIVbits.PLLPOST = 1;
    __builtin_write_OSCCONH(0x01); // switch to FRC PLL
    __builtin_write_OSCCONL(0x01);

	// set up timer 0 for Timer_Delay_ms_escape
	T1CONbits.TCKPS = 3; // divided by 256;

    // indicate that bootloader has started with short LED chaser
    init_leds();
	rapid_flash(200);

    InitSDCardDetect();
	boot_address_cache[0] = read_program_word(0);
	boot_address_cache[1] = read_program_word(2);

	// Main bootloader loop.
	while(1) {
		// Check to see if an SD card is inserted, if not break out and boot main program (if present; otherwise bootloader starts again)
		memoryCardSystemUp = !PORTFbits.RF4;
		if( !memoryCardSystemUp )
			break;
		// OK, turn power on and try mounting it
		SetSDCardPower(1);
		error = init_sdcard_and_open_hex_file();
		if( error ) {
			if( error == 3 ) // no hex file detected, bail out and try booting program
				break;
		Failure:
			error_flash(1, error, &memoryCardSystemUp, 0);
			while( PORTFbits.RF4 )
				;
		} else {
            last_bytes_written = 0;
            progress[0] = 0;
            progress[1] = 0;
			// OK, now let's see if it's a valid HEX file and compare it to what's already in flash memory
			ret = check_hex_file(&cur_file, 1);
			if( ret == difference ) {
				// It's a valid HEX file and its contents are different from what's in flash so let's reprogram the chip
				if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK ) {
					error = 3;
					goto Failure;
				}
                total_bytes_to_write = last_bytes_written;
                last_bytes_written = 0;
				reflash_from_hex_file(&cur_file, 2);
				// Now re-open the HEX file so we can verify that it was successfully reprogrammed with no errors
				if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK ) {
					error = 3;
					goto Failure;
				}	
                last_bytes_written = 0;
				ret = check_hex_file(&cur_file, 3);
                set_led_state(0, 0);
			}
			if( ret == no_difference ) {
				// Either the flash didn't need updating or it was updated successfully; break out and boot the main program
				break;
			} else {
				// After re-flash, file contents still differ from what's in flash; this does not normally happen and we should indicate an error.
				error_flash(1, 3 + ret, &memoryCardSystemUp, 0);
				while( PORTFbits.RF4 )
					;
			}
		}
	}
    set_led_state(0, 0);
	boot_main_program();

	return 0;
}
